#!/usr/bin/python3
# ==============================================================================================
# FSK   design
# ==============================================================================================
class Fsk:
    """RTTY_FSK.Fsk Class: Generates an FSK audio stream from any text"""
    import os
    def __init__(self, baud=50, size=5, stop=1.5, mark=1275, space=1445, dbg=False):
        self._dbg = dbg
        self._running = False
        self._exists = True
        self._opened = False
        self._baud = baud
        self._size = size
        self._stop = stop
        self._mark = mark
        self._space = space
        self._duration = 0
        self._exconv = False
        self._noconv = False
        self.player = "minimodem"
        self.pid = None
        self.p = 0
    # ------------------------------------------------------------------------------------------
    def __repr__(self):
        return "player:'{}' baud:{} size:{} stop:{} mark:{} space:{}".format(self.player, self._baud, self._size, self._stop, self._mark, self._space)
    # ------------------------------------------------------------------------------------------
    @property
    def dbg(self):
        return self._dbg
    @dbg.setter
    def dbg(self, x):
        self._dbg = x
    # ------------------------------------------------------------------------------------------
    @property
    def opened(self):
        return self._opened
    # ------------------------------------------------------------------------------------------
    @property
    def running(self):
        return self._running
    # ------------------------------------------------------------------------------------------
    @property
    def exists(self):
        return self._exists
    # ------------------------------------------------------------------------------------------
    @property
    def baud(self):
        return self._baud
    @baud.setter
    def baud(self, x):
        self._baud = x
    # ------------------------------------------------------------------------------------------
    @property
    def size(self):
        return self._size
    @size.setter
    def size(self, x):
        self._size = x
    # ------------------------------------------------------------------------------------------
    @property
    def stop(self):
        return self._stop
    @stop.setter
    def stop(self, x):
        self._stop = x
    # ------------------------------------------------------------------------------------------
    @property
    def parity(self):
        return self._parity
    @parity.setter
    def parity(self, x):
        self._parity = x
    # ------------------------------------------------------------------------------------------
    @property
    def mark(self):
        return self._mark
    @mark.setter
    def mark(self, x):
        self._mark = x
    # ------------------------------------------------------------------------------------------
    @property
    def space(self):
        return self._space
    @space.setter
    def space(self, x):
        self._space = x
    # ------------------------------------------------------------------------------------------
    @property
    def ita2extern(self):
        return self._noconv
    @ita2extern.setter
    def ita2extern(self, x):
        self._noconv = False
        if (x == True) and (self._exconv): self._noconv=True
    # ------------------------------------------------------------------------------------------
    @property
    def player(self):
        return self._player
    @player.setter
    def player(self, x):
        import os,sys,subprocess
        self._exists = True
        try:
            output = subprocess.check_output(x+ " -V 2>/dev/null", shell=True)
        except Exception as e:
            self._player = ""
            self._exists = False
            sys.tracebacklimit = None
            raise ValueError("FSK modem '"+x+"' not found")
        self._exconv = False
        if (b'RxControl') in output:
            self._exconv = True
            self._noconv = True
        self._player = x
    # ------------------------------------------------------------------------------------------
    def Close(self):
        if (self._opened):
            try:
                self._opened = False
                self._running = False
                self.p.terminate()[0]
                self.Wait()
            except:
                pass
    # ------------------------------------------------------------------------------------------
    def Open(self):
        self._opened = False
        self._running = False
        if (self._exists):
            try:
                import os, sys, subprocess
                from subprocess import Popen, PIPE
                if (self._noconv): s = "-n"
                else:		       s = "-q"
                self.p=subprocess.Popen([self.player,"-A","-q",s,
                                                     "--tx-carrier",
                                                     "--stopbits", str(self.stop),
                                                     "-"+str(self.size),
                                                     "-M", str(self.mark), "-S", str(self.space),
                                                     "--tx", str(self.baud)],
                                                     bufsize=-1,
                                                     preexec_fn=os.setsid, stdin=subprocess.PIPE, shell=False)
                self.pid = self.p.pid
                self._duration = (1/self.baud)*(1+self.size+self.stop+1)
                self._opened = True
            except Exception as e:
                pass
    # ------------------------------------------------------------------------------------------
    # No idea on how to wait for the stdin data fully processed by the minimodem processs
    # So, we wait for the known processing delay with option to quicky exit if process is killed
    # ------------------------------------------------------------------------------------------
    def Print(self, name):
        if (self._opened):
            try:
                import time
                if (self._dbg): print("INFO:  FSK    PRINT STARTED")
                self._running = True
                self.p.stdin.write(name.encode())
                self.p.stdin.flush()
                delay=self._duration*len(name)
                if (delay <= 0.25):
                    time.sleep(delay)
                else:
                    loop = int(delay/0.25)+1
                    for i in range(loop):
                        time.sleep(0.25)
                        if (not self._running): break
                if (self._dbg): print("INFO:  FSK    PRINT ENDED")
            except Exception as e:
                self._running = False
                if (self._dbg): print("ERROR: FSK    PRINT NOT STARTED ({0})".format(e))
                pass
    # ------------------------------------------------------------------------------------------
    def Abort(self):            # Process to be killed has been detached from console (os.setsid)
        import signal, os       # To avoid killing father process
        if (self._running):	    
            try:
              # self.p.stdin.close()
                self._running = False
                os.killpg(os.getpgid(self.pid), signal.SIGTERM)
                self.Wait()
                if (self._dbg): print("INFO:  FSK    PRINT KILLED")
            except Exception as e:
                if (self._dbg): print("ERROR: FSK    PRINT NOT KILLED ({0})".format(e))
                pass
    # ------------------------------------------------------------------------------------------
    def KillConsole(self):      # Killing from direct console launch
        import os
        if (self._opened):
            self.p.stdin.close()
            self._running = False
          # os.killpg(os.getpgid(self.pid), signal.SIGTERM)
            os.system('stty sane')
    # ------------------------------------------------------------------------------------------
    def Wait(self):
        self.p.wait()
    # ------------------------------------------------------------------------------------------
    def Test(self):			# Minimodem takes care of the characters encoding
        import time
        import RTTY_ITA2
        print("TEST:  FSK GENERATOR")
        msg1 = "THEQUICKBROWNFOXJUMPSOVERTHEMAZYDOG 0123456789"
        msg2 = "RYRYRYRYRYRYRYRYRYRYRYRYRYRYRYRYRYRYRYRYRYRYRY"
        ita2 = RTTY_ITA2.Ita2()
        try:
            self.ita2extern = True
            if (self.ita2extern):
                msg1 = ita2.toIta2(msg1)
                msg2 = ita2.toIta2(msg2)
                print("         Modified minimodem detected")
                print("         External ITA2 conversion")
            else:
                print("         Internal ITA2 conversion")
            self.Open()
            self.Print(msg1)
            print("TEST:  SLEEPING 2s")
            time.sleep(2)
            print("TEST:  RESTART")
            self.Print(msg2)
            self.Close()
        except:
            print("-> Not passed")
            pass
# ==============================================================================================
if __name__ == '__main__':
    import sys
    import signal
    from inspect import getdoc
    def signal_term_handler(signal, frame):
        this.KillConsole()
        sys.exit(0)
    signal.signal(signal.SIGINT,  signal_term_handler)
    signal.signal(signal.SIGTERM, signal_term_handler)
    print(getdoc(Fsk))
    this = Fsk()
    this.dbg = True
    this.Test()
# ==============================================================================================
