#!/usr/bin/python3
# ==============================================================================================
# ITA2 converter design
# ==============================================================================================
class Ita2:
    """RTTY_Ita2.Ita2 Class: Translates string to and from ITA2"""
    # ------------------------------------------------------------------------------------------
    #           ' ' ,'!' ,'"' ,'#' ,'$' ,'%' ,'&' ,'\'' ,'(',')' ,'*' ,'+' ,',' ,'-' ,'.' ,'/' ,
    #           '0' ,'1' ,'2' ,'3' ,'4' ,'5' ,'6' ,'7' ,'8' ,'9' ,':' ,';' ,'<' ,'=' ,'>' ,'?' ,
    #           '@' ,'A' ,'B' ,'C' ,'D' ,'E' ,'F' ,'G' ,'H' ,'I' ,'J' ,'K' ,'L' ,'M' ,'N' ,'O' ,
    #           'P' ,'Q' ,'R' ,'S' ,'T' ,'U' ,'V' ,'W' ,'X' ,'Y' ,'Z' ,'[' ,'\\',']' ,'^' ,'_' ,
    #           '`' ,'a' ,'b' ,'c' ,'d' ,'e' ,'f' ,'g' ,'h' ,'i' ,'j' ,'k' ,'l' ,'m' ,'n' ,'o' ,
    #           'p' ,'q' ,'r ','s' ,'t' ,'u' ,'v' ,'w' ,'x' ,'y' ,'z' ,'{' ,'|' ,'}' ,'~' ,' '
    # ------------------------------------------------------------------------------------------
    UsAsc2Ita=[0x04,0x8D,0x91,0x04,0x94,0x04,0x9A,0x85,0x8F,0x92,0x04,0x91,0x8C,0x83,0x9C,0x9D,
               0x96,0x97,0x93,0x81,0x8A,0x90,0x95,0x87,0x86,0x98,0x8E,0x9E,0x8F,0x04,0x92,0x99,
               0x04,0x03,0x19,0x0E,0x09,0x01,0x0D,0x1A,0x14,0x06,0x0B,0x0F,0x12,0x1C,0x0C,0x18,
               0x16,0x17,0x0A,0x05,0x10,0x07,0x1E,0x13,0x1D,0x15,0x11,0x8F,0x9D,0x92,0x04,0x83,
               0x8B,0x03,0x19,0x0E,0x09,0x01,0x0D,0x1A,0x14,0x06,0x0B,0x0F,0x12,0x1C,0x0C,0x18,
               0x16,0x17,0x0A,0x05,0x10,0x07,0x1E,0x13,0x1D,0x15,0x11,0x8F,0x04,0x92,0x00,0x00]
    UsIta2Asc=[[''  ,''  ],['E' ,'3' ],['\n','\n'],['A' ,'-' ],[' ' ,' ' ],['S' ,'\''],['I' ,'8' ],['U' ,'7' ],
               ['\r','\r'],['D' ,''  ],['R' ,'4' ],['J' ,''  ],['N' ,',' ],['F' ,'!' ],['C' ,':' ],['K' ,'(' ],
               ['T' ,'5' ],['Z' ,'+' ],['L' ,')' ],['W' ,'2' ],['H' ,'£' ],['Y' ,'6' ],['P' ,'0' ],['Q' ,'1' ],
               ['O' ,'9' ],['B' ,'?' ],['G' ,'&' ],[''  ,''  ],['M' ,'.' ],['X' ,'/' ],['V' ,';' ],[''  ,''  ]]
    # ------------------------------------------------------------------------------------------
    FrAsc2Ita=[0x04,0x04,0x91,0x04,0x94,0x9A,0x04,0x85,0x8F,0x92,0x04,0x91,0x8C,0x83,0x9C,0x9D,
               0x96,0x97,0x93,0x81,0x8A,0x90,0x95,0x87,0x86,0x98,0x8E,0x04,0x8F,0x9E,0x92,0x99,
               0x04,0x03,0x19,0x0E,0x09,0x01,0x0D,0x1A,0x14,0x06,0x0B,0x0F,0x12,0x1C,0x0C,0x18,
               0x16,0x17,0x0A,0x05,0x10,0x07,0x1E,0x13,0x1D,0x15,0x11,0x8F,0x9D,0x92,0x04,0x83,
               0x8B,0x03,0x19,0x0E,0x09,0x01,0x0D,0x1A,0x14,0x06,0x0B,0x0F,0x12,0x1C,0x0C,0x18,
               0x16,0x17,0x0A,0x05,0x10,0x07,0x1E,0x13,0x1D,0x15,0x11,0x8F,0x04,0x92,0x00,0x00]
    FrIta2Asc=[[''  ,''  ],['E' ,'3' ],['\n','\n'],['A' ,'-' ],[' ' ,' ' ],['S' ,'\''],['I' ,'8' ],['U' ,'7' ],
               ['\r','\r'],['D' ,''  ],['R' ,'4' ],['J' ,''  ],['N' ,',' ],['F' ,'é' ],['C' ,':' ],['K' ,'(' ],
               ['T' ,'5' ],['Z' ,'+' ],['L' ,')' ],['W' ,'2' ],['H' ,'h' ],['Y' ,'6' ],['P' ,'0' ],['Q' ,'1' ],
               ['O' ,'9' ],['B' ,'?' ],['G' ,'%' ],[''  ,''  ],['M' ,'.' ],['X' ,'/' ],['V' ,'=' ],[''  ,''  ]]
    # ------------------------------------------------------------------------------------------
    Asc2Ita=[UsAsc2Ita,FrAsc2Ita]
    Ita2Asc=[UsIta2Asc,FrIta2Asc]
    # ------------------------------------------------------------------------------------------
    FIGS =0x1b
    LTRS =0x1f
    # ------------------------------------------------------------------------------------------
    def __init__(self,table=0,padding=True,dbg=False):
        self._dbg    = dbg        
        self._padding=padding            # Set NULL padding if true
        self._table  =table              # Set US convert table if 0
        self._figure =-1                 # State of Letter / Figures
    # ------------------------------------------------------------------------------------------
    def __repr__(self):
        return "table:'{}' padding:'{}'".format(self._table,self_padding)
    # ------------------------------------------------------------------------------------------
    @property
    def dbg(self):
        return self._dbg
    @dbg.setter
    def dbg(self,x):
        self._dbg=x
    # ------------------------------------------------------------------------------------------
    @property
    def padding(self):
        return self._padding
    @padding.setter
    def padding(self,x):
        self._padding=x
    # ------------------------------------------------------------------------------------------
    @property
    def table(self):
        return self._table
    @table.setter
    def table(self,x):
        if (x>1): x=1
        self._table=x
    # ------------------------------------------------------------------------------------------
    def ita2prettyprint(self,ita2):
        msg=""
        for i in ita2:
            if   (ord(i)==self.LTRS): 
                msg=msg+" Lt "
            elif (ord(i)==self.FIGS): 
                msg=msg+" Fi "
            else: 
                msg=msg+" {0:#04x}".format(ord(i)&0x7f)
        return msg
    # ------------------------------------------------------------------------------------------
    def crlf(self):
        return self.translate("\r\n")
    # ------------------------------------------------------------------------------------------
    def null(self):
        return chr(0x00)+chr(0x00)
    # ------------------------------------------------------------------------------------------
    def toIta2(self, text):
        ita2 = ""
        # --------------------------------------------------------------------------------------
        if (self._dbg ): print("Input:  "+text)
        if (self._dbg ): print("        "+"".join("{0:#04x}".format(ord(c)).replace("0x","") for c in text))
        #---------------------------------------------------------------------------------------
        for char in text:
            # ----------------------------------------------------------------------------------
            if (char == '\r'):
                ita2+=chr(0x08)
            if (char == '\n'):
                ita2+=chr(0x02)
            # ----------------------------------------------------------------------------------
            # Taking care of the LETTER/FIGURE changes
            # ----------------------------------------------------------------------------------
            if (ord(char) >=0x20) and (ord(char) <=0x7F):
               #tmp = self.UsAsc2Ita[ord(char)-0x20]
                tmp = self.Asc2Ita[self._table][ord(char)-0x20]
                if (tmp<0x80):
                    if ((self._figure==-1) or (self._figure==1)):
                        ita2+=chr(self.LTRS)
                        self._figure=0
                if (tmp>=0x80):
                    if ((self._figure==-1) or (self._figure==0)):
                        ita2+=chr(self.FIGS)
                        self._figure=1
                    tmp-=0x80
                ita2+=chr(tmp)
            # ----------------------------------------------------------------------------------
        if (self._dbg ): print("Output: "+"".join("{0:#04x}".format(ord(i)).replace("0x","") for i in ita2))
        # --------------------------------------------------------------------------------------
        return ita2
    # ------------------------------------------------------------------------------------------
    def toAscii(self, text):
        asc  = ""
        self._figure=0
        if (self._dbg ): print("Input:  "+"".join("{0:#04x}".format(ord(i)).replace("0x","") for i in text))
        #---------------------------------------------------------------------------------------
        for char in text:
            if   (char == chr(self.LTRS)): 
               #if (self._dbg ): print("LETTER")
                self._figure = 0
            elif (char == chr(self.FIGS)):
               #if (self._dbg ): print("FIGURE")
                self._figure = 1
            else:
                asc+=self.Ita2Asc[self._table][ord(char)][self._figure]
        # --------------------------------------------------------------------------------------
        if (self._dbg ): print("Output: "+"".join("{0:#04x}".format(ord(c)&0x7f).replace("0x","") for c in asc))
        if (self._dbg ): print("        "+" ".join(c for c in asc))
        return asc
    # ------------------------------------------------------------------------------------------
    def Test(self):
        print("TEST:  ITA2 Conversions")
        try:
            for i in range(len(self.Asc2Ita)):
                this.table=i
                if (self._dbg): print("----")
                if (self._dbg): print("TEST:  ASCII to ITA2  CONVERT TEST - Table N°"+str(i))
                ita=self.toIta2 ("THEQUICKBROWNFOXJUMPSOVERTHELAZYDOG0123456789 ()?!/.'+-=;&%")
                if (self._dbg): print("")
                if (self._dbg): print("TEST:  ITA2  to ASCII CONVERT TEST - Table N°"+str(i))
                asc=self.toAscii(ita)
        except:
            print("-> Not passed")                
            pass 
# ==============================================================================================
if __name__ == '__main__':
    import sys
    import signal
    from inspect import getdoc
    def signal_term_handler(signal, frame):
        sys.exit(0)
    signal.signal(signal.SIGINT,  signal_term_handler)
    signal.signal(signal.SIGTERM, signal_term_handler)
    print(getdoc(Ita2))
    this=Ita2()
    this.dbg=True
    this.Test()
# ==============================================================================================
